<?php
/**
 * SVG icons related functions and filters
 *
 * @package     Sinatra
 * @author      Sinatra Team <hello@sinatrawp.com>
 * @since       1.0.0
 */

/**
 * Do not allow direct script access.
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Filters the arguments for a single nav menu item to include dropdown indicators.
 *
 * @since 1.2.0
 *
 * @param stdClass $args  An object of wp_nav_menu() arguments.
 * @param WP_Post  $item  Menu item data object.
 * @param int      $depth Depth of menu item. Used for padding.
 */
function sinatra_nav_menu_item_args( $args, $item, $depth ) {

	if ( sinatra_option( 'main_nav_sub_indicators' ) ) {
		$dropdown_indicator = sinatra()->icons->get_svg( 'chevron-down' );

		if ( in_array( 'menu-item-has-children', $item->classes, true ) ) {
			if ( false === strpos( $args->link_after, $dropdown_indicator ) ) {
				$args->link_after .= $dropdown_indicator;
			}
		} else {
			$args->link_after = str_replace( $dropdown_indicator, '', $args->link_after );
		}
	}

	return $args;
}
add_filter( 'nav_menu_item_args', 'sinatra_nav_menu_item_args', 10, 3 );


/**
 * Display social icons in social links menu.
 *
 * @param  string  $item_output The menu item output.
 * @param  WP_Post $item        Menu item object.
 * @param  int     $depth       Depth of the menu.
 * @param  array   $args        wp_nav_menu() arguments.
 * @return string  $item_output The menu item output with social icon.
 */
function sinatra_nav_menu_social_icons( $item_output, $item, $depth, $args ) {

	// Get supported social icons.
	$social_icons = sinatra_social_links_icons();

	// Change SVG icon inside social links menu if there is supported URL.
	if ( false !== strpos( $args->menu_class, 'sinatra-socials-menu' ) ) {

		foreach ( $social_icons as $attr => $value ) {
			if ( false !== strpos( $item_output, $attr ) ) {
				$item_output = str_replace(
					$args->link_after,
					'</span>' . sinatra()->icons->get_svg( $value, array( 'aria-hidden' => 'true' ) ) . sinatra()->icons->get_svg(
						$value,
						array(
							'class'       => 'bottom-icon',
							'aria-hidden' => 'true',
						)
					),
					$item_output
				);
			}
		}
	}

	return $item_output;
}
add_filter( 'walker_nav_menu_start_el', 'sinatra_nav_menu_social_icons', 10, 4 );

/**
 * Returns an array of supported social links (URL and icon name).
 *
 * @return array $social_links_icons
 */
function sinatra_social_links_icons() {

	// Supported social links icons.
	$social_links_icons = array(
		'500px.com'       => '500px',
		'amazon.com'      => 'amazon',
		'behance.net'     => 'behance',
		'digg.com'        => 'digg',
		'dribbble.com'    => 'dribbble',
		'deviantart'      => 'deviantart',
		'etsy.com'        => 'etsy',
		'facebook.com'    => 'facebook',
		'flipboard.com'   => 'flipboard',
		'flickr.com'      => 'flickr',
		'foursquare.com'  => 'foursquare',
		'github.com'      => 'github',
		'plus.google.com' => 'google-plus',
		'instagram.com'   => 'instagram',
		'linkedin.com'    => 'linkedin',
		'mailto:'         => 'mail',
		'medium.com'      => 'medium',
		'pinterest.com'   => 'pinterest',
		'reddit.com'      => 'reddit',
		'skype.com'       => 'skype',
		'skype:'          => 'skype',
		'snapchat.com'    => 'snapchat',
		'soundcloud.com'  => 'soundcloud',
		'spotify.com'     => 'spotify',
		'tumblr.com'      => 'tumblr',
		'twitch.tv'       => 'twitch',
		'twitter.com'     => 'twitter',
		'vimeo.com'       => 'vimeo',
		'xing.com'        => 'xing',
		'vk.com'          => 'vkontakte',
		'youtube.com'     => 'youtube',
		'yelp.com'        => 'yelp',
		'wa.me'           => 'whatsapp',
	);

	/**
	 * Filter Sinatra social links icons.
	 *
	 * @since 1.0.0
	 * @param array $social_links_icons Array of social links icons.
	 */
	return apply_filters( 'sinatra_social_links_icons', $social_links_icons );
}

if ( ! class_exists( 'Sinatra_Icons' ) ) :
	/**
	 * Sinatra Icons class.
	 *
	 * @since 1.2.0
	 */
	class Sinatra_Icons {
		/**
		 * Primary class constructor.
		 *
		 * @since 1.2.0
		 */
		public function __construct() {
		}

		/**
		 * GET SVG CODE
		 * Get the SVG code for the specified icon
		 *
		 * @param string $icon Icon name.
		 * @param array  $args Parameters needed to display an SVG.
		 */
		public function get_svg( $icon = '', $args = array() ) {

			$arr = self::get();

			if ( ! array_key_exists( $icon, $arr ) ) {
				return null;
			}

			$args = wp_parse_args(
				$args,
				array(
					'class' => '',
				)
			);

			// .si-icon is a required class.
			if ( false === strpos( $args['class'], 'si-icon' ) ) {
				$args['class'] = trim( 'si-icon ' . $args['class'] );
			}

			$repl = '<svg ';

			if ( ! empty( $args ) ) {
				foreach ( $args as $key => $value ) {
					if ( ! empty( $value ) ) {
						$repl .= esc_html( $key ) . '="' . esc_attr( $value ) . '" ';
					}
				}
			}

			$svg = preg_replace( '/^<svg /', $repl, trim( $arr[ $icon ] ) ); // Add extra attributes to SVG code.
			$svg = preg_replace( "/([\n\t]+)/", ' ', $svg ); // Remove newlines & tabs.
			$svg = preg_replace( '/>\s*</', '><', $svg );    // Remove whitespace between SVG tags.

			// Make sure that only our allowed tags and attributes are included.
			$svg = wp_kses(
				$svg,
				array(
					'svg'     => array(
						'class'       => true,
						'xmlns'       => true,
						'width'       => true,
						'height'      => true,
						'viewbox'     => true,
						'aria-hidden' => true,
						'aria-label'  => true,
						'role'        => true,
						'focusable'   => true,
					),
					'path'    => array(
						'fill'      => true,
						'fill-rule' => true,
						'd'         => true,
						'transform' => true,
					),
					'polygon' => array(
						'fill'      => true,
						'fill-rule' => true,
						'points'    => true,
						'transform' => true,
						'focusable' => true,
					),
				)
			);

			if ( ! $svg ) {
				return null;
			}

			return $svg;
		}

		/**
		 * Get SVG icons array.
		 *
		 * @return array
		 */
		public function get() {
			return apply_filters( 'sinatra_icons_svg', self::$icons );
		}

		/**
		 * Get all available SVG icons
		 *
		 * @return array
		 */
		public function get_all_svg_icons() {

			$arr = self::get();

			$return = array();

			foreach ( $arr as $icon => $svg ) {

				$svg = self::get_svg( $icon );

				if ( $svg ) {
					$return[ $icon ] = self::get_svg( $icon );
				}
			}

			return $return;
		}

		/**
		 * Get icon for post entry meta.
		 *
		 * @since  1.1.0
		 * @param  string $slug    Icon slug.
		 * @param  string $icon    Icon HTML markup.
		 * @param  string $post_id Current post ID.
		 * @return string          Icon HTML markup.
		 */
		public function get_meta_icon( $slug = '', $icon = '', $post_id = '' ) {

			$return = '';

			if ( is_single( $post_id ) ) {
				if ( sinatra_option( 'single_entry_meta_icons' ) ) {
					$return = $icon;
				}
			} elseif ( ! is_single() ) {
				if ( sinatra_option( 'entry_meta_icons' ) ) {
					$return = $icon;
				}
			}

			return apply_filters( 'sinatra_' . $slug . '_meta_icon', $return, $post_id );
		}

		/**
		 * Icons.
		 * Store the code for all SVGs in an array.
		 *
		 * @var array
		 */
		public static $icons = array(
			'500px'          => '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M7.912 21.451c0 0.032 0.16 0.48 0.256 0.705 0.512 1.249 1.281 2.37 2.242 3.331s2.082 1.73 3.331 2.242c1.281 0.545 2.659 0.833 4.068 0.833s2.787-0.288 4.068-0.833c1.249-0.512 2.37-1.281 3.331-2.242s1.73-2.082 2.242-3.331c0.545-1.281 0.833-2.659 0.833-4.068s-0.288-2.787-0.833-4.068c-0.512-1.249-1.281-2.37-2.242-3.331s-2.082-1.73-3.331-2.242c-1.281-0.545-2.659-0.833-4.068-0.833-1.441 0-2.851 0.288-4.132 0.833-1.025 0.416-2.755 1.537-3.748 2.595v0-8.68h14.51c0.512 0 0.512-0.737 0.512-0.993s0-0.993-0.545-0.993l-15.631 0.032c-0.416 0-0.705 0.352-0.705 0.673v12.14c0 0.384 0.48 0.673 0.929 0.769 0.897 0.192 1.089-0.096 1.313-0.384l0.032-0.032c0.32-0.48 1.377-1.537 1.377-1.537 1.602-1.602 3.78-2.498 6.086-2.498s4.452 0.897 6.054 2.498c1.634 1.634 2.498 3.78 2.498 6.054s-0.897 4.42-2.498 6.054c-1.602 1.602-3.812 2.498-6.086 2.498-1.538 0-3.043-0.416-4.324-1.185v-7.399c0-0.993 0.416-2.050 1.153-2.851 0.833-0.929 1.954-1.441 3.171-1.441 1.185 0 2.306 0.448 3.139 1.281 0.833 0.801 1.281 1.89 1.281 3.043 0 2.466-1.954 4.388-4.42 4.388-0.48 0-1.345-0.224-1.377-0.224-0.512-0.16-0.705 0.545-0.769 0.769-0.256 0.865 0.128 1.057 0.224 1.089 0.801 0.256 1.313 0.288 2.018 0.288 3.491 0 6.342-2.851 6.342-6.342 0-3.459-2.851-6.31-6.342-6.31-1.698 0-3.331 0.641-4.516 1.858-1.153 1.121-1.826 2.627-1.826 4.132v0.032c0 0.192 0 4.613 0 6.054v0c-0.673-0.737-1.313-1.858-1.73-2.979-0.16-0.448-0.545-0.352-1.057-0.192-0.288 0.096-0.993 0.288-0.833 0.801v0zM15.375 19.721c0 0.224 0.192 0.384 0.32 0.512v0.032c0.192 0.192 0.384 0.288 0.577 0.288 0.16 0 0.224-0.064 0.256-0.096 0.096-0.096 1.089-1.089 1.185-1.185l1.121 1.089c0.096 0.128 0.224 0.16 0.352 0.192 0.192 0 0.384-0.096 0.577-0.32 0.48-0.48 0.224-0.737 0.128-0.865l-1.121-1.121 1.153-1.185c0.256-0.288 0.032-0.577-0.192-0.801-0.32-0.32-0.641-0.416-0.833-0.224l-1.153 1.153-1.185-1.185c-0.064-0.064-0.128-0.096-0.224-0.096-0.16 0-0.352 0.096-0.545 0.32-0.352 0.352-0.448 0.609-0.256 0.801l1.185 1.185-1.153 1.185c-0.16 0.096-0.192 0.224-0.192 0.32zM17.938 3.834c-1.89 0-3.876 0.384-5.349 1.025-0.16 0.064-0.256 0.192-0.256 0.352-0.032 0.16 0.032 0.384 0.128 0.673 0.096 0.224 0.32 0.865 0.801 0.673 1.505-0.577 3.171-0.897 4.677-0.897 1.73 0 3.363 0.32 4.933 0.993 1.249 0.513 2.402 1.281 3.684 2.402 0.096 0.096 0.192 0.128 0.288 0.128 0.256 0 0.48-0.256 0.705-0.48 0.352-0.384 0.577-0.705 0.224-1.025-1.217-1.153-2.563-2.018-4.196-2.723-1.794-0.737-3.684-1.121-5.637-1.121zM28.027 27.025v0c-0.224-0.224-0.416-0.352-0.577-0.416s-0.32 0-0.448 0.096l-0.096 0.128c-1.153 1.153-2.53 2.082-4.036 2.723-1.57 0.673-3.235 0.993-4.933 0.993s-3.363-0.32-4.933-0.993c-1.505-0.641-2.851-1.538-4.036-2.723-1.217-1.217-2.114-2.563-2.723-4.036-0.577-1.409-0.769-2.498-0.833-2.851 0-0.032 0-0.064 0-0.064-0.064-0.416-0.448-0.448-0.993-0.352-0.224 0.032-0.929 0.128-0.865 0.641v0c0.192 1.153 0.512 2.274 0.961 3.363 0.737 1.73 1.794 3.299 3.107 4.613 1.345 1.345 2.883 2.402 4.612 3.107 1.794 0.769 3.716 1.153 5.67 1.153s3.876-0.384 5.67-1.153c1.73-0.737 3.299-1.794 4.613-3.107 0 0 0.064-0.064 0.128-0.128 0.192-0.128 0.32-0.416-0.288-0.993z"></path></svg>',
			'amazon'         => '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M27.724 27.076c0.536-0.268 0.947 0.143 0.393 0.858s-5.005 4.79-12.476 4.79-13.191-5.112-14.943-7.221c-0.483-0.554 0.072-0.804 0.393-0.59 5.237 3.182 13.424 8.419 26.633 2.163zM31.424 25.020c0.268 0.358 0 1.93-0.465 3.074-0.465 1.126-1.144 1.913-1.519 2.216-0.393 0.322-0.679 0.197-0.465-0.268s1.376-3.325 0.912-3.932c-0.465-0.59-2.645-0.304-3.432-0.232-0.769 0.072-0.929 0.143-1.001-0.018-0.161-0.411 1.555-1.108 2.681-1.251 1.126-0.125 2.931-0.054 3.289 0.411zM24.381 17.102c0 1.966 2.306 3.772 2.306 3.772l-4.058 4.004c-1.591-1.501-2.788-2.753-2.788-2.753-0.179-0.179-0.322-0.393-0.447-0.59-3.235 5.058-13.12 4.737-13.12-3.092 0-7.293 8.633-8.276 12.119-8.401v-2.27c0-0.483 0.179-2.681-2.538-2.681 0 0-2.717 0-3.879 3.539l-5.255-0.483c0-3.521 3.343-7.454 9.634-7.454 6.274 0 8.026 4.075 8.026 5.881v10.528zM12.369 17.477c0 3.629 6.024 4.487 6.024-1.233v-2.896c-2.413 0.072-6.024 0.751-6.024 4.129z"></path></svg>',
			'arrow-left'     => '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M25.358 14.087h-15.482l7.074-7.074c0.534-0.534 0.534-1.335 0-1.868s-1.335-0.534-1.868 0l-9.342 9.342c-0.133 0.133-0.267 0.267-0.267 0.4-0.133 0.267-0.133 0.667 0 1.068 0.133 0.133 0.133 0.267 0.267 0.4l9.342 9.342c0.267 0.267 0.667 0.4 0.934 0.4s0.667-0.133 0.934-0.4c0.534-0.534 0.534-1.335 0-1.868l-7.074-7.074h15.482c0.801 0 1.335-0.534 1.335-1.335s-0.534-1.335-1.335-1.335z"></path></svg>',
			'arrow-right'    => '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M26.559 15.955c0.133-0.267 0.133-0.667 0-1.068-0.133-0.133-0.133-0.267-0.267-0.4l-9.342-9.342c-0.534-0.534-1.335-0.534-1.868 0s-0.534 1.335 0 1.868l7.074 7.074h-15.482c-0.801 0-1.335 0.534-1.335 1.335s0.534 1.335 1.335 1.335h15.482l-7.074 7.074c-0.534 0.534-0.534 1.335 0 1.868 0.267 0.267 0.667 0.4 0.934 0.4s0.667-0.133 0.934-0.4l9.342-9.342c0.133-0.133 0.267-0.267 0.267-0.4z"></path></svg>',
			'behance'        => '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M28.841 9.666h-8.008v-1.877h8.008zM15.578 17.236c0.626 0.876 0.938 1.877 0.938 3.128 0 1.001-0.188 1.877-0.563 2.565-0.438 0.751-0.938 1.314-1.564 1.814-0.688 0.438-1.439 0.751-2.252 1.001-0.813 0.188-1.689 0.313-2.565 0.313h-9.572v-19.894h9.259c0.938 0 1.814 0.188 2.628 0.375 0.751 0.188 1.376 0.5 1.939 0.876s1.001 0.938 1.314 1.564c0.313 0.563 0.438 1.376 0.438 2.315s-0.188 1.814-0.688 2.44c-0.438 0.688-1.126 1.251-2.002 1.627 1.251 0.375 2.127 1.001 2.69 1.877zM4.192 14.17h4.504c0.813 0 1.439-0.063 1.939-0.438s0.751-0.876 0.751-1.752c0-0.438-0.063-0.813-0.25-1.126-0.188-0.25-0.375-0.501-0.688-0.688-0.25-0.125-0.563-0.313-0.938-0.375s-0.751-0.125-1.126-0.125h-4.192zM12.199 20.113c0-1.001-0.313-1.689-0.876-2.127s-1.314-0.626-2.252-0.626h-4.88v5.443h4.755c0.438 0 0.876-0.063 1.251-0.125 0.375-0.125 0.688-0.25 1.001-0.438s0.563-0.438 0.688-0.813c0.188-0.313 0.313-0.751 0.313-1.314zM32.031 19.926h-10.323c0 1.001 0.375 2.19 1.001 2.753 0.563 0.563 1.439 0.876 2.502 0.876 0.813 0 1.501-0.25 2.065-0.688 0.563-0.375 0.876-0.688 1.064-1.314h3.441c-0.563 1.939-1.439 3.003-2.565 3.754s-2.502 1.189-4.129 1.189c-1.126 0-2.127-0.188-3.066-0.563-0.876-0.375-1.627-0.876-2.252-1.501-0.626-0.688-1.126-1.501-1.501-2.44-0.313-0.876-0.5-1.877-0.5-3.003 0-1.064 0.188-2.065 0.563-3.003 0.313-0.938 0.813-1.689 1.439-2.377 0.688-0.688 1.439-1.189 2.315-1.627 0.938-0.375 1.939-0.563 3.003-0.563 1.189 0 2.252 0.25 3.191 0.688 0.876 0.501 1.627 1.126 2.19 1.877 0.626 0.813 1.001 1.752 1.251 2.753s0.375 2.252 0.313 3.191zM28.090 17.361c-0.125-0.938-0.438-1.689-0.876-2.19-0.5-0.563-1.314-0.813-2.252-0.813-0.626 0-1.126 0.125-1.564 0.375-0.375 0.188-0.751 0.438-1.001 0.751s-0.375 0.626-0.5 1.001c-0.125 0.313-0.188 0.563-0.188 0.876z"></path></svg>',
			'bookmark'       => '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M22.689 2.075h-13.346c-2.269 0-4.004 1.735-4.004 4.004v21.354c0 0.534 0.267 0.934 0.667 1.201s0.934 0.133 1.335-0.133l8.542-6.139 8.542 6.139c0.267 0.133 0.534 0.267 0.801 0.267s0.4 0 0.667-0.133c0.4-0.267 0.667-0.667 0.667-1.201v-21.354c0.133-2.269-1.602-4.004-3.87-4.004zM24.023 24.897l-7.207-5.205c-0.267-0.133-0.534-0.267-0.801-0.267s-0.534 0.133-0.801 0.267l-7.207 5.205v-18.818c0-0.801 0.534-1.335 1.335-1.335h13.346c0.801 0 1.335 0.534 1.335 1.335v18.818z"></path></svg>',
			'chat'           => '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 576 512"><path d="M512 160h-96V64c0-35.3-28.7-64-64-64H64C28.7 0 0 28.7 0 64v160c0 35.3 28.7 64 64 64h32v52c0 7.1 5.8 12 12 12 2.4 0 4.9-.7 7.1-2.4L224 288h128c35.3 0 64-28.7 64-64v-32h96c17.6 0 32 14.4 32 32v160c0 17.6-14.4 32-32 32h-64v49.6l-80.2-45.4-7.3-4.2H256c-17.6 0-32-14.4-32-32v-96l-32 18.1V384c0 35.3 28.7 64 64 64h96l108.9 61.6c2.2 1.6 4.7 2.4 7.1 2.4 6.2 0 12-4.9 12-12v-52h32c35.3 0 64-28.7 64-64V224c0-35.3-28.7-64-64-64zm-128 64c0 17.6-14.4 32-32 32H215.6l-7.3 4.2-80.3 45.4V256H64c-17.6 0-32-14.4-32-32V64c0-17.6 14.4-32 32-32h288c17.6 0 32 14.4 32 32v160z"/></svg>',
			'chevron-down'   => '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M24.958 10.483c-0.534-0.534-1.335-0.534-1.868 0l-7.074 7.074-7.074-7.074c-0.534-0.534-1.335-0.534-1.868 0s-0.534 1.335 0 1.868l8.008 8.008c0.267 0.267 0.667 0.4 0.934 0.4s0.667-0.133 0.934-0.4l8.008-8.008c0.534-0.534 0.534-1.335 0-1.868z"></path></svg>',
			'chevron-up'     => '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M24.958 18.491l-8.008-8.008c-0.534-0.534-1.335-0.534-1.868 0l-8.008 8.008c-0.534 0.534-0.534 1.335 0 1.868s1.335 0.534 1.868 0l7.074-7.074 7.074 7.074c0.267 0.267 0.667 0.4 0.934 0.4s0.667-0.133 0.934-0.4c0.534-0.534 0.534-1.335 0-1.868z"></path></svg>',
			'clock'          => '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M16.016 0.74c-8.141 0-14.681 6.54-14.681 14.681s6.54 14.681 14.681 14.681 14.681-6.54 14.681-14.681-6.54-14.681-14.681-14.681zM16.016 27.433c-6.673 0-12.012-5.339-12.012-12.012s5.339-12.012 12.012-12.012c6.673 0 12.012 5.339 12.012 12.012s-5.339 12.012-12.012 12.012zM21.888 16.889l-4.538-2.269v-7.207c0-0.801-0.534-1.335-1.335-1.335s-1.335 0.534-1.335 1.335v8.008c0 0.534 0.267 0.934 0.801 1.201l5.339 2.669c0.133 0.133 0.267 0.133 0.534 0.133 0.534 0 0.934-0.267 1.201-0.801s0-1.335-0.667-1.735z"></path></svg>',
			'deviantart'     => '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M30.217 17.548c-1.501-1.814-3.378-2.94-5.505-3.816-2.753-1.126-5.63-1.564-8.571-1.627-0.188 0-0.438 0-0.751 0 0-1.001 0-1.939 0-3.003-1.314 0.125-2.565 0.313-4.192 0.438 0 0.188 0 0.25 0 0.375 0 0.751 0 1.564 0 2.315 0.063 0.25-0.063 0.375-0.313 0.438-2.127 0.438-4.129 1.126-6.006 2.252-1.439 0.876-2.69 1.939-3.629 3.378-1.126 1.689-1.564 3.566-1.064 5.505 5.13 0 10.072 0 15.202 0 0-3.191 0-6.131 0-9.196 0.626 0 1.314 0 1.939 0 1.627 0.125 3.253 0.375 4.755 1.001 2.002 0.938 3.378 2.377 3.941 4.504 0.063 0.125 0.063 0.5 0.125 0.5-1.814 0-3.378 0-5.318 0 0 0 0-0.313 0-0.438 0-1.189 0-2.44 0-3.691 0-0.25-0.063-0.375-0.313-0.375-0.813-0.188-1.501-0.313-2.315-0.438-0.5-0.063-1.251-0.063-1.564-0.125 0 2.815 0 5.693 0 8.258 5.13 0 10.072 0 15.202 0 0.063-0.626 0.188-0.938 0.188-1.376 0.063-1.877-0.626-3.441-1.814-4.88zM11.198 20.614c-1.939 0-3.503 0-5.255 0 0.125-2.252 2.69-5.067 5.255-5.38 0 1.752 0 3.441 0 5.38z"></path></svg>',
			'digg'           => '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M9.634 11.292h3.191v10.886h-3.191zM5.13 6.788h3.191v15.39h-8.321v-10.886h5.13zM5.13 13.857h-1.939v5.756h1.939zM9.634 10.041h3.191v-3.253h-3.191zM14.076 11.292h8.321v14.764h-8.321v-2.565h5.13v-1.314h-5.13zM17.267 19.613h1.939v-5.756h-1.939zM32.031 11.292v14.764h-8.321v-2.565h5.13v-1.314h-5.13v-10.886zM28.841 13.857h-1.939v5.756h1.939z"></path></svg>',
			'dribbble'       => '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M15.953 0.532c-8.821 0-16.016 7.195-16.016 16.016s7.195 16.016 16.016 16.016c8.884 0 16.016-7.195 16.016-16.016s-7.132-16.016-16.016-16.016zM26.588 7.914c1.877 2.315 3.066 5.255 3.066 8.508-0.438-0.125-4.942-1.001-9.509-0.438-0.125-0.25-0.188-0.501-0.313-0.688-0.25-0.688-0.563-1.376-0.876-2.002 5.005-2.065 7.32-5.005 7.632-5.38zM15.953 2.847c3.503 0 6.694 1.314 9.071 3.441-0.188 0.375-2.252 3.128-7.132 4.942-2.252-4.066-4.755-7.445-5.13-8.008 1.064-0.25 2.127-0.375 3.191-0.375zM10.197 4.16c0.313 0.5 2.753 3.879 5.067 7.945-6.444 1.689-12.074 1.627-12.7 1.627 0.938-4.254 3.754-7.758 7.632-9.572zM2.315 16.547c0-0.125 0-0.25 0-0.438 0.563 0.063 7.195 0.125 14.076-1.939 0.375 0.751 0.751 1.564 1.064 2.315-0.188 0.063-0.313 0.125-0.5 0.188-7.069 2.252-10.823 8.508-11.136 9.009-2.19-2.44-3.503-5.63-3.503-9.134zM15.953 30.248c-3.128 0-6.068-1.126-8.383-2.878 0.25-0.563 3.003-5.881 10.761-8.571 0 0 0.063 0 0.063-0.063 1.939 5.005 2.753 9.196 2.94 10.385-1.627 0.688-3.441 1.126-5.38 1.126zM23.648 27.871c-0.188-0.813-0.876-4.817-2.69-9.76 4.317-0.688 8.070 0.438 8.508 0.626-0.563 3.816-2.753 7.069-5.818 9.134z"></path></svg>',
			'edit-3'         => '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M4.004 23.429h5.339c0.4 0 0.667-0.133 0.934-0.4l14.681-14.681c0.534-0.534 0.534-1.335 0-1.868l-5.339-5.339c-0.534-0.534-1.335-0.534-1.868 0l-14.681 14.681c-0.267 0.267-0.4 0.534-0.4 0.934v5.339c0 0.801 0.534 1.335 1.335 1.335zM5.339 17.29l13.346-13.346 3.47 3.47-13.346 13.346h-3.47v-3.47zM28.027 27.433h-24.023c-0.801 0-1.335 0.534-1.335 1.335s0.534 1.335 1.335 1.335h24.023c0.801 0 1.335-0.534 1.335-1.335s-0.534-1.335-1.335-1.335z"></path></svg>',
			'etsy'           => '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M3.503 1.658c-0.063-0.438 0.125-0.751 0.751-0.751 0.563 0 6.631 0.188 10.698 0.188 8.383 0 10.448-0.688 11.887-0.688 0.751 0.063 0.938 0.626 0.938 1.001s-0.375 1.126-0.375 2.127c0 1.001 0.25 2.002 0.25 2.94 0 0.876-0.751 1.126-1.189 1.126-0.375 0-0.626-0.125-0.626-0.876-0.063-1.439 1.001-3.503-9.259-3.503-5.193 0-5.318 0.626-5.318 2.252 0 1.689 0 7.132 0 7.945 0 0.876 0.25 1.126 0.938 1.126s6.006 0 7.445 0c1.439 0 2.94-0.751 3.128-2.815 0.063-1.064 0.313-1.126 0.876-1.126s1.001 0.438 0.938 1.439c-0.125 1.064-0.313 2.127-0.313 3.566 0 1.501 0.313 2.94 0.313 4.004s-1.439 1.314-1.689 0.125c-0.25-1.064-0.688-3.003-3.941-3.003s-5.881 0-6.631 0c-0.751 0-1.064 0.438-1.064 1.627s0 7.007 0 8.446c0 1.501 0.438 3.066 3.378 3.066 3.003 0 8.383 0.313 10.886-1.627 1.001-0.813 1.376-2.065 1.689-4.066 0.188-0.876 2.19-0.876 2.065 1.251-0.063 1.001-1.001 3.066-1.001 4.442 0 1.064 0.25 2.628-0.751 2.628-0.876 0-0.5-0.5-4.379-0.5s-17.204 0.125-19.456 0.125c-0.813 0-1.001-0.25-1.001-0.751s0.313-0.876 1.001-0.876c0.688 0 3.691 0 3.691-3.754 0-3.816 0-17.642 0-19.707 0-2.002-0.438-4.066-2.94-4.317-1.001-0.125-0.938-0.5-0.938-1.064z"></path></svg>',
			'external-link'  => '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M24.023 15.421c-0.801 0-1.335 0.534-1.335 1.335v8.008c0 0.801-0.534 1.335-1.335 1.335h-14.681c-0.801 0-1.335-0.534-1.335-1.335v-14.681c0-0.801 0.534-1.335 1.335-1.335h8.008c0.801 0 1.335-0.534 1.335-1.335s-0.534-1.335-1.335-1.335h-8.008c-2.269 0-4.004 1.735-4.004 4.004v14.681c0 2.269 1.735 4.004 4.004 4.004h14.681c2.269 0 4.004-1.735 4.004-4.004v-8.008c0-0.801-0.534-1.335-1.335-1.335zM29.229 2.876c-0.133-0.267-0.4-0.534-0.667-0.667-0.133-0.133-0.4-0.133-0.534-0.133h-8.008c-0.801 0-1.335 0.534-1.335 1.335s0.534 1.335 1.335 1.335h4.805l-12.412 12.412c-0.534 0.534-0.534 1.335 0 1.868 0.267 0.267 0.534 0.4 0.934 0.4s0.667-0.133 0.934-0.4l12.412-12.412v4.805c0 0.801 0.534 1.335 1.335 1.335s1.335-0.534 1.335-1.335v-8.008c0-0.133 0-0.4-0.133-0.534z"></path></svg>',
			'facebook'       => '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M18.518 32.438v-14.639h4.88l0.751-5.693h-5.631v-3.629c0-1.627 0.438-2.753 2.815-2.753h3.003v-5.067c-0.5-0.125-2.315-0.25-4.379-0.25-4.379 0-7.32 2.628-7.32 7.507v4.192h-4.942v5.693h4.942v14.639z"></path></svg>',
			'flipboard'      => '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M4.317 2.721v23.398h7.82v-7.82h7.758v-7.758h7.82v-7.82z"></path></svg>',
			'flickr'         => '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M7.257 23.679c-4.004 0-7.257-3.253-7.257-7.257s3.253-7.257 7.257-7.257c4.004 0 7.257 3.253 7.257 7.257s-3.253 7.257-7.257 7.257zM24.774 23.679c-4.004 0-7.257-3.253-7.257-7.257s3.253-7.257 7.257-7.257c4.004 0 7.257 3.253 7.257 7.257s-3.253 7.257-7.257 7.257z"></path></svg>',
			'foursquare'     => '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M24.649 0.407c0 0-14.702 0-17.079 0-2.315 0-3.003 1.752-3.003 2.878s0 27.151 0 27.151c0 1.251 0.626 1.752 1.001 1.877 0.438 0.188 1.439 0.313 2.065-0.438 0 0 8.070-9.384 8.195-9.509 0.25-0.188 0.25-0.188 0.438-0.188 0.438 0 3.566 0 5.255 0 2.19 0 2.502-1.564 2.753-2.502 0.188-0.751 2.315-11.699 3.066-15.202 0.5-2.628-0.125-4.066-2.69-4.066zM24.274 19.675c0.188-0.751 2.315-11.699 3.066-15.202zM23.648 5.036l-0.688 3.754c-0.125 0.375-0.626 0.813-1.064 0.813-0.5 0-6.694 0-6.694 0-0.751 0-1.251 0.501-1.251 1.251v0.813c0 0.751 0.501 1.251 1.251 1.251 0 0 5.13 0 5.693 0 0.5 0 1.001 0.626 0.938 1.189-0.125 0.563-0.688 3.378-0.751 3.691-0.063 0.25-0.375 0.813-1.064 0.813-0.5 0-4.567 0-4.567 0-0.876 0-1.126 0.125-1.689 0.813s-5.568 6.757-5.568 6.757c-0.063 0.063-0.125 0.063-0.125 0v-21.208c0-0.5 0.438-1.064 1.064-1.064 0 0 13.138 0 13.701 0 0.5 0 0.938 0.501 0.813 1.126z"></path></svg>',
			'github'         => '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M30.28 11.543c0-2.252-0.751-4.129-2.127-5.568 0.25-0.5 0.938-2.628-0.188-5.505 0 0-1.752-0.563-5.693 2.127-1.689-0.438-3.441-0.688-5.255-0.688-1.752 0-3.503 0.25-5.193 0.688-4.004-2.69-5.693-2.127-5.693-2.127-1.189 2.878-0.438 5.005-0.25 5.505-1.314 1.439-2.127 3.316-2.127 5.568 0 8.008 4.88 9.76 9.509 10.323-0.626 0.5-1.189 1.439-1.376 2.753-1.189 0.563-4.192 1.439-6.006-1.752 0 0-1.126-1.939-3.191-2.127 0 0-2.065 0-0.188 1.251 0 0 1.376 0.688 2.315 3.066 0 0 1.251 3.691 7.007 2.44 0.063 1.752 0.063 4.442 0.063 4.942h10.323c0-0.688 0-4.004 0-6.757 0-1.939-0.626-3.191-1.376-3.879 4.63-0.5 9.447-2.252 9.447-10.26zM4.129 21.74c-0.063 0.063-0.25 0.125-0.375 0.063s-0.25-0.25-0.188-0.313c0.063-0.125 0.188-0.188 0.375-0.063 0.125 0.063 0.188 0.188 0.188 0.313zM4.942 22.678c-0.063 0.063-0.313 0-0.438-0.125s-0.125-0.313-0.063-0.438c0.125-0.063 0.313-0.063 0.438 0.125 0.125 0.125 0.188 0.313 0.063 0.438zM5.756 23.867c-0.125 0.063-0.313 0-0.438-0.188s-0.125-0.438 0-0.5c0.125-0.063 0.313 0 0.438 0.188s0.125 0.375 0 0.5zM6.882 24.993c-0.125 0.125-0.375 0.125-0.5-0.063-0.188-0.188-0.25-0.375-0.125-0.5 0.125-0.188 0.375-0.125 0.5 0.063 0.188 0.125 0.25 0.375 0.125 0.5zM8.446 25.681c-0.063 0.188-0.313 0.25-0.5 0.188-0.25-0.063-0.438-0.25-0.375-0.438s0.313-0.25 0.563-0.188c0.188 0.063 0.375 0.25 0.313 0.438zM9.697 26.119c-0.25 0-0.438-0.125-0.438-0.313s0.188-0.313 0.438-0.313c0.25 0 0.438 0.125 0.438 0.313s-0.188 0.313-0.438 0.313zM11.324 25.932c-0.25 0.063-0.438-0.063-0.501-0.25 0-0.125 0.188-0.313 0.438-0.375s0.438 0.063 0.501 0.25c0 0.125-0.188 0.313-0.438 0.375z"></path></svg>',
			'google-plus'    => '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="41" height="32" viewBox="0 0 41 32"><path d="M25.686 17.012c0 7.489-5.023 12.798-12.584 12.798-7.239 0-13.102-5.863-13.102-13.102s5.863-13.102 13.102-13.102c3.539 0 6.489 1.287 8.776 3.432l-3.557 3.414c-0.965-0.929-2.663-2.020-5.219-2.020-4.469 0-8.115 3.7-8.115 8.276s3.646 8.276 8.115 8.276c5.184 0 7.132-3.736 7.436-5.648h-7.436v-4.504h12.369c0.125 0.661 0.215 1.323 0.215 2.181zM41.183 14.831v3.754h-3.736v3.736h-3.754v-3.736h-3.736v-3.754h3.736v-3.736h3.754v3.736h3.736z"></path></svg>',
			'instagram'      => '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M16.016 3.284c4.317 0 4.755 0.063 6.444 0.125 1.627 0.063 2.44 0.313 3.003 0.5 0.751 0.313 1.314 0.688 1.814 1.189 0.563 0.563 0.938 1.126 1.189 1.814 0.25 0.626 0.5 1.439 0.563 3.003 0.063 1.752 0.125 2.19 0.125 6.506s-0.063 4.755-0.125 6.444c-0.063 1.627-0.313 2.44-0.5 3.003-0.313 0.751-0.688 1.314-1.189 1.814-0.563 0.563-1.126 0.938-1.814 1.189-0.626 0.25-1.439 0.5-3.003 0.563-1.752 0.063-2.19 0.125-6.506 0.125s-4.755-0.063-6.444-0.125c-1.627-0.063-2.44-0.313-3.003-0.5-0.751-0.313-1.314-0.688-1.814-1.189-0.563-0.563-0.938-1.126-1.189-1.814-0.25-0.626-0.5-1.439-0.563-3.003-0.063-1.752-0.125-2.19-0.125-6.506s0.063-4.755 0.125-6.444c0.063-1.627 0.313-2.44 0.5-3.003 0.313-0.751 0.688-1.314 1.189-1.814 0.563-0.563 1.126-0.938 1.814-1.189 0.626-0.25 1.439-0.5 3.003-0.563 1.752-0.063 2.19-0.125 6.506-0.125M16.016 0.407c-4.379 0-4.88 0.063-6.569 0.125-1.752 0.063-2.94 0.313-3.879 0.688-1.064 0.438-2.002 1.001-2.878 1.877s-1.439 1.814-1.877 2.878c-0.375 1.001-0.626 2.127-0.688 3.879-0.063 1.689-0.125 2.19-0.125 6.569s0.063 4.88 0.125 6.569c0.063 1.752 0.313 2.94 0.688 3.879 0.438 1.064 1.001 2.002 1.877 2.878s1.814 1.439 2.878 1.877c1.001 0.375 2.127 0.626 3.879 0.688 1.689 0.063 2.19 0.125 6.569 0.125s4.88-0.063 6.569-0.125c1.752-0.063 2.94-0.313 3.879-0.688 1.064-0.438 2.002-1.001 2.878-1.877s1.439-1.814 1.877-2.878c0.375-1.001 0.626-2.127 0.688-3.879 0.063-1.689 0.125-2.19 0.125-6.569s-0.063-4.88-0.125-6.569c-0.063-1.752-0.313-2.94-0.688-3.879-0.438-1.064-1.001-2.002-1.877-2.878s-1.814-1.439-2.878-1.877c-1.001-0.375-2.127-0.626-3.879-0.688-1.689-0.063-2.19-0.125-6.569-0.125zM16.016 8.164c-4.567 0-8.258 3.691-8.258 8.258s3.691 8.258 8.258 8.258c4.567 0 8.258-3.691 8.258-8.258s-3.691-8.258-8.258-8.258zM16.016 21.803c-2.94 0-5.38-2.44-5.38-5.38s2.44-5.38 5.38-5.38c2.94 0 5.38 2.44 5.38 5.38s-2.44 5.38-5.38 5.38zM26.463 7.851c0 1.064-0.813 1.939-1.877 1.939s-1.939-0.876-1.939-1.939c0-1.064 0.876-1.877 1.939-1.877s1.877 0.813 1.877 1.877z"></path></svg>',
			'linkedin'       => '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M23.961 16.798c-0.438-0.5-1.189-0.751-2.252-0.751-1.376 0-2.315 0.438-2.878 1.189-0.563 0.813-0.813 1.939-0.813 3.378v10.948c0 0.188-0.063 0.375-0.25 0.563-0.125 0.125-0.375 0.25-0.563 0.25h-5.756c-0.188 0-0.375-0.125-0.563-0.25-0.125-0.188-0.25-0.375-0.25-0.563v-20.207c0-0.188 0.125-0.375 0.25-0.563 0.188-0.125 0.375-0.25 0.563-0.25h5.568c0.188 0 0.375 0.063 0.5 0.125s0.188 0.25 0.25 0.438c0 0.188 0.063 0.375 0.063 0.438s0 0.25 0 0.501c1.439-1.376 3.378-2.002 5.693-2.002 2.69 0 4.755 0.626 6.256 1.939 1.501 1.376 2.252 3.316 2.252 5.943v13.638c0 0.188-0.063 0.375-0.25 0.563-0.125 0.125-0.375 0.25-0.563 0.25h-5.881c-0.188 0-0.375-0.125-0.563-0.25-0.125-0.188-0.25-0.375-0.25-0.563v-12.325c0-1.064-0.188-1.877-0.563-2.44zM6.631 7.101c-0.751 0.751-1.689 1.126-2.753 1.126s-2.002-0.375-2.753-1.126c-0.751-0.751-1.126-1.627-1.126-2.753 0-1.064 0.375-1.939 1.126-2.69s1.689-1.189 2.753-1.189c1.064 0 2.002 0.438 2.753 1.189s1.126 1.627 1.126 2.69c0 1.126-0.375 2.002-1.126 2.753zM7.57 11.355v20.207c0 0.188-0.125 0.375-0.25 0.563-0.188 0.125-0.375 0.25-0.563 0.25h-5.756c-0.188 0-0.375-0.125-0.563-0.25-0.125-0.188-0.25-0.375-0.25-0.563v-20.207c0-0.188 0.125-0.375 0.25-0.563 0.188-0.125 0.375-0.25 0.563-0.25h5.756c0.188 0 0.375 0.125 0.563 0.25 0.125 0.188 0.25 0.375 0.25 0.563z"></path></svg>',
			'mail'           => '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M26.693 3.41h-21.354c-2.269 0-4.004 1.735-4.004 4.004v16.016c0 2.269 1.735 4.004 4.004 4.004h21.354c2.269 0 4.004-1.735 4.004-4.004v-16.016c0-2.269-1.735-4.004-4.004-4.004zM5.339 6.079h21.354c0.534 0 0.934 0.267 1.201 0.801l-11.878 8.275-11.878-8.275c0.267-0.534 0.667-0.801 1.201-0.801zM26.693 24.764h-21.354c-0.801 0-1.335-0.534-1.335-1.335v-13.48l11.211 7.874c0.267 0.133 0.534 0.267 0.801 0.267s0.534-0.133 0.801-0.267l11.211-7.874v13.48c0 0.801-0.534 1.335-1.335 1.335z"></path></svg>',
			'medium'         => '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M31.781 7.726l-10.135-5.067c-0.063 0-0.063 0-0.125 0-0.125 0-0.313 0.063-0.375 0.125l-6.194 10.197 6.944 11.324 10.010-16.203c0.063-0.125 0-0.313-0.125-0.375zM11.636 9.415v10.948l9.76 4.88zM22.209 25.619l8.946 4.504c0.438 0.25 0.876 0 0.876-0.563v-19.894zM10.51 7.789l-9.322-4.63-0.626-0.375c-0.063 0-0.125-0.063-0.188-0.063-0.125 0-0.188 0.063-0.25 0.125s-0.125 0.188-0.125 0.375v21.333c0 0.375 0.25 0.813 0.563 0.938l9.196 4.63c0.125 0.063 0.25 0.125 0.375 0.125 0.313 0 0.501-0.25 0.501-0.688v-21.584c0-0.063 0-0.125-0.125-0.188z"></path></svg>',
			'message-square' => '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M25.358 2.075h-18.685c-2.269 0-4.004 1.735-4.004 4.004v21.354c0 0.534 0.267 1.068 0.801 1.201 0.133 0.133 0.4 0.133 0.534 0.133 0.4 0 0.667-0.133 0.934-0.4l4.938-4.938h15.482c2.269 0 4.004-1.735 4.004-4.004v-13.346c0-2.269-1.735-4.004-4.004-4.004zM26.693 19.425c0 0.801-0.534 1.335-1.335 1.335h-16.016c-0.4 0-0.667 0.133-0.934 0.4l-3.070 3.070v-18.151c0-0.801 0.534-1.335 1.335-1.335h18.685c0.801 0 1.335 0.534 1.335 1.335v13.346z"></path></svg>',
			'pinterest'      => '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M10.635 12.669c0-1.376 0.313-2.502 1.064-3.441 0.688-0.938 1.564-1.439 2.565-1.439 0.813 0 1.439 0.25 1.877 0.813s0.688 1.251 0.688 2.065c0 0.501-0.125 1.126-0.313 1.814-0.188 0.751-0.375 1.627-0.688 2.565-0.313 1.001-0.563 1.752-0.688 2.315-0.25 1.001-0.063 1.877 0.563 2.628 0.626 0.688 1.439 1.064 2.44 1.064 1.752 0 3.191-1.001 4.317-2.94 1.126-2.002 1.689-4.379 1.689-7.195 0-2.127-0.688-3.879-2.065-5.255-1.376-1.314-3.316-2.002-5.818-2.002-2.815 0-5.067 0.876-6.757 2.69-1.752 1.752-2.628 3.879-2.628 6.381 0 1.501 0.438 2.753 1.251 3.754 0.313 0.313 0.375 0.688 0.313 1.064-0.125 0.313-0.25 0.813-0.375 1.501-0.063 0.25-0.188 0.438-0.375 0.5s-0.375 0.063-0.563 0c-1.314-0.563-2.252-1.439-2.94-2.753s-1.001-2.815-1.001-4.504c0-1.126 0.188-2.19 0.563-3.316s0.876-2.19 1.627-3.191c0.751-1.064 1.689-1.939 2.69-2.753 1.064-0.813 2.315-1.439 3.879-1.939 1.501-0.438 3.128-0.688 4.817-0.688 1.814 0 3.441 0.313 4.942 0.938 1.501 0.563 2.815 1.376 3.816 2.377 1.064 1.064 1.814 2.19 2.44 3.503 0.563 1.314 0.876 2.69 0.876 4.066 0 3.754-0.938 6.882-2.878 9.322-1.939 2.502-4.379 3.691-7.382 3.691-1.001 0-1.939-0.188-2.815-0.688-0.876-0.438-1.501-1.001-1.877-1.689-0.688 2.94-1.126 4.692-1.314 5.255-0.375 1.439-1.251 3.191-2.69 5.255h-1.314c-0.25-2.565-0.188-4.692 0.188-6.381l2.44-10.323c-0.375-0.813-0.563-1.814-0.563-3.066z"></path></svg>',
			'play'           => '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M26.025 14.354l-18.685-12.012c-0.4-0.267-0.934-0.267-1.335 0-0.4 0.133-0.667 0.534-0.667 1.068v24.023c0 0.534 0.267 0.934 0.667 1.201 0.267 0.133 0.4 0.133 0.667 0.133s0.534-0.133 0.667-0.267l18.685-12.012c0.4-0.267 0.667-0.667 0.667-1.068s-0.267-0.934-0.667-1.068zM8.008 25.031v-19.219l14.948 9.609-14.948 9.609z"></path></svg>',
			'reddit'         => '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M32.031 16.172c0-1.752-1.376-3.128-3.128-3.128-1.001 0-1.877 0.438-2.44 1.189-2.502-1.814-6.006-2.94-9.947-3.003l2.315-6.882 5.756 1.376c0.125 0 0.188 0 0.25 0 0.063 1.314 1.126 2.377 2.502 2.377s2.44-1.126 2.44-2.502c0-1.376-1.064-2.44-2.44-2.44-1.064 0-1.877 0.563-2.315 1.439 0 0-0.063-0.063-0.125-0.063l-6.319-1.501c-0.313-0.063-0.626 0.125-0.688 0.375l-2.69 7.82c-3.816 0.125-7.195 1.189-9.697 2.94-0.563-0.688-1.376-1.126-2.377-1.126-1.752 0-3.128 1.376-3.128 3.128 0 1.251 0.751 2.377 1.877 2.878-0.125 0.438-0.188 0.938-0.188 1.439 0 5.13 6.381 9.322 14.264 9.322s14.264-4.192 14.264-9.322c0-0.438-0.063-0.938-0.188-1.376 1.189-0.5 2.002-1.627 2.002-2.94zM8.696 18.674c0-1.251 1.064-2.315 2.315-2.315s2.315 1.064 2.315 2.315c0 1.251-1.064 2.315-2.315 2.315s-2.315-1.064-2.315-2.315zM21.208 25.056c-1.064 1.064-2.753 1.564-5.193 1.564s-4.129-0.5-5.193-1.564c-0.125-0.125-0.125-0.375 0-0.563 0.125-0.125 0.375-0.125 0.501 0 0.938 0.938 2.44 1.376 4.692 1.376 2.19 0 3.754-0.438 4.692-1.376 0.125-0.125 0.375-0.125 0.5 0 0.125 0.188 0.125 0.438 0 0.563zM21.083 20.989c-1.314 0-2.377-1.001-2.377-2.315 0-1.251 1.064-2.315 2.377-2.315 1.251 0 2.252 1.064 2.252 2.315 0 1.314-1.001 2.315-2.252 2.315z"></path></svg>',
			'search'         => '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M28.962 26.499l-4.938-4.938c1.602-2.002 2.669-4.671 2.669-7.474 0-6.673-5.339-12.012-12.012-12.012s-12.012 5.339-12.012 12.012c0 6.673 5.339 12.012 12.012 12.012 2.803 0 5.472-0.934 7.474-2.669l4.938 4.938c0.267 0.267 0.667 0.4 0.934 0.4s0.667-0.133 0.934-0.4c0.534-0.534 0.534-1.335 0-1.868zM5.339 14.087c0-5.205 4.137-9.342 9.342-9.342s9.342 4.137 9.342 9.342c0 2.536-1.068 4.938-2.669 6.54 0 0 0 0 0 0s0 0 0 0c-1.735 1.735-4.004 2.669-6.54 2.669-5.339 0.133-9.476-4.004-9.476-9.209z"></path></svg>',
			'share-2'        => '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M24.023 19.425c-1.468 0-2.803 0.667-3.737 1.602l-7.074-4.137c0-0.534 0.133-0.934 0.133-1.468s-0.133-0.934-0.267-1.468l7.074-4.137c1.068 0.934 2.402 1.602 3.87 1.602 2.936 0 5.339-2.402 5.339-5.339s-2.402-5.339-5.339-5.339-5.339 2.402-5.339 5.339c0 0.534 0.133 0.934 0.267 1.468l-7.207 4.137c-0.934-0.934-2.269-1.602-3.737-1.602-2.936 0-5.339 2.402-5.339 5.339s2.402 5.339 5.339 5.339c1.468 0 2.803-0.667 3.737-1.602l7.074 4.137c0 0.534-0.133 0.934-0.133 1.468 0 2.936 2.402 5.339 5.339 5.339s5.339-2.402 5.339-5.339-2.402-5.339-5.339-5.339zM24.023 3.41c1.468 0 2.669 1.201 2.669 2.669s-1.201 2.669-2.669 2.669-2.669-1.201-2.669-2.669 1.201-2.669 2.669-2.669zM8.008 18.091c-1.468 0-2.669-1.201-2.669-2.669s1.201-2.669 2.669-2.669 2.669 1.201 2.669 2.669c0 1.468-1.201 2.669-2.669 2.669zM24.023 27.433c-1.468 0-2.669-1.201-2.669-2.669 0-0.534 0.133-0.934 0.4-1.335 0 0 0 0 0 0s0 0 0 0c0.4-0.801 1.335-1.335 2.269-1.335 1.468 0 2.669 1.201 2.669 2.669s-1.201 2.669-2.669 2.669z"></path></svg>',
			'shopping-cart'  => '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M14.681 27.433c0 1.474-1.195 2.669-2.669 2.669s-2.669-1.195-2.669-2.669c0-1.474 1.195-2.669 2.669-2.669s2.669 1.195 2.669 2.669zM29.362 27.433c0 1.474-1.195 2.669-2.669 2.669s-2.669-1.195-2.669-2.669c0-1.474 1.195-2.669 2.669-2.669s2.669 1.195 2.669 2.669zM31.764 6.613c-0.267-0.4-0.667-0.534-1.068-0.534h-21.621l-1.068-5.605c-0.133-0.667-0.667-1.068-1.335-1.068h-5.339c-0.801 0-1.335 0.534-1.335 1.335s0.534 1.335 1.335 1.335h4.271l1.068 5.605c0 0 0 0.133 0 0.133l2.269 11.077c0.4 1.868 2.002 3.203 3.87 3.203 0 0 0 0 0.133 0h12.946c2.002 0 3.604-1.335 4.004-3.203l2.135-11.211c0-0.4 0-0.801-0.267-1.068zM27.227 18.358c-0.133 0.667-0.667 1.068-1.335 1.068h-12.946c-0.667 0-1.201-0.4-1.335-1.068l-2.002-9.609h19.486l-1.868 9.609z"></path></svg>',
			'skype'          => '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M30.905 19.801c0.25-1.001 0.313-2.127 0.313-3.191 0-8.321-6.694-15.077-15.015-15.077-0.876 0-1.752 0.063-2.565 0.188-1.376-0.813-2.94-1.314-4.692-1.314-4.817 0-8.696 3.941-8.696 8.759 0 1.564 0.438 3.128 1.189 4.379-0.188 1.001-0.313 2.002-0.313 3.065 0 8.321 6.757 15.015 15.077 15.015 0.938 0 1.814-0.063 2.753-0.25 1.189 0.688 2.628 1.064 4.129 1.064 4.817 0 8.696-3.941 8.696-8.759 0-1.376-0.313-2.69-0.876-3.879zM23.773 23.742c-0.688 1.001-1.689 1.752-3.066 2.315-1.314 0.563-2.878 0.813-4.692 0.813-2.127 0-3.941-0.375-5.38-1.126-1.001-0.5-1.877-1.251-2.502-2.19s-0.938-1.814-0.938-2.69c0-0.563 0.188-1.064 0.626-1.439s0.938-0.563 1.564-0.563c0.501 0 0.938 0.125 1.314 0.438s0.688 0.751 0.876 1.314c0.313 0.563 0.563 1.126 0.876 1.501s0.751 0.751 1.251 1.001c0.563 0.25 1.314 0.375 2.19 0.375 1.251 0 2.315-0.25 3.066-0.813 0.751-0.5 1.126-1.126 1.126-1.877 0-0.563-0.188-1.064-0.626-1.439-0.375-0.375-0.938-0.626-1.564-0.876-0.688-0.188-1.627-0.438-2.753-0.688-1.564-0.313-2.94-0.688-4.004-1.189-1.064-0.438-1.939-1.064-2.565-1.877s-1.001-1.814-1.001-3.003c0-1.126 0.375-2.19 1.064-3.066s1.627-1.564 2.94-2.002c1.251-0.5 2.69-0.688 4.379-0.688 1.376 0 2.565 0.125 3.503 0.438 1.001 0.313 1.877 0.751 2.502 1.251 0.688 0.501 1.189 1.064 1.501 1.627s0.438 1.189 0.438 1.752c0 0.501-0.188 1.001-0.626 1.439-0.375 0.438-0.938 0.626-1.564 0.626-0.5 0-0.938-0.125-1.251-0.375s-0.563-0.688-0.938-1.189c-0.375-0.751-0.813-1.314-1.314-1.689s-1.376-0.563-2.502-0.563c-1.126 0-2.002 0.188-2.628 0.626-0.626 0.375-0.938 0.876-0.938 1.439 0 0.313 0.063 0.626 0.313 0.876 0.188 0.25 0.501 0.501 0.876 0.688s0.813 0.375 1.189 0.438c0.438 0.125 1.126 0.313 2.127 0.563 1.189 0.25 2.315 0.501 3.316 0.813 1.001 0.375 1.877 0.751 2.565 1.189 0.751 0.501 1.314 1.064 1.689 1.814 0.438 0.688 0.626 1.627 0.626 2.628 0 1.251-0.375 2.377-1.064 3.378z"></path></svg>',
			'snapchat'       => '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="30" height="32" viewBox="0 0 30 32"><path d="M15.158 2.981c3.217-0.036 5.881 1.77 7.221 4.665 0.411 0.876 0.483 2.234 0.483 3.2 0 1.144-0.089 2.27-0.161 3.414 0.143 0.072 0.34 0.125 0.5 0.125 0.643 0 1.18-0.483 1.823-0.483 0.608 0 1.484 0.429 1.484 1.144 0 1.716-3.593 1.394-3.593 2.896 0 0.268 0.107 0.518 0.215 0.769 0.858 1.877 2.485 3.682 4.415 4.487 0.465 0.197 0.929 0.304 1.43 0.411 0.322 0.072 0.5 0.304 0.5 0.626 0 1.215-3.092 1.716-3.915 1.841-0.358 0.554-0.089 1.859-1.037 1.859-0.733 0-1.466-0.232-2.252-0.232-0.375 0-0.751 0.018-1.108 0.089-2.127 0.358-2.842 2.645-6.328 2.645-3.36 0-4.165-2.288-6.238-2.645-0.375-0.072-0.751-0.089-1.126-0.089-0.804 0-1.573 0.268-2.216 0.268-1.001 0-0.697-1.323-1.072-1.895-0.822-0.125-3.915-0.626-3.915-1.841 0-0.322 0.179-0.554 0.5-0.626 0.5-0.107 0.965-0.215 1.43-0.411 1.913-0.787 3.575-2.61 4.415-4.487 0.107-0.25 0.215-0.5 0.215-0.769 0-1.501-3.611-1.215-3.611-2.878 0-0.697 0.822-1.144 1.448-1.144 0.554 0 1.108 0.465 1.805 0.465 0.197 0 0.393-0.036 0.572-0.125-0.072-1.126-0.161-2.252-0.161-3.396 0-0.965 0.072-2.342 0.483-3.217 1.573-3.396 4.236-4.63 7.793-4.665z"></path></svg>',
			'soundcloud'     => '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M0.626 21.552h-0.626v-3.879h0.626zM2.252 22.804h-0.626v-6.068h0.626zM3.503 23.179h-0.626v-7.069h0.626zM4.817 23.492h-0.626v-7.069h0.626zM6.381 23.492h-0.626v-9.322h0.626zM7.695 23.492h-0.626v-10.573h0.626zM9.259 23.492h-0.626v-11.261h0.626zM10.573 23.492h-0.626v-11.574h0.626zM12.199 23.492h-0.688v-11.261h0.688zM13.451 23.492h-0.626v-10.886h0.626zM14.764 23.492h-0.688v-12.199h0.688zM16.328 23.492h-0.626v-13.138h0.626zM28.090 23.367c-0.063 0-10.948 0-10.948 0-0.25 0-0.5-0.188-0.5-0.438v-12.512c0-0.25 0.125-0.375 0.438-0.501 0.751-0.313 1.627-0.438 2.502-0.438 3.629 0 6.631 2.753 6.944 6.319 0.5-0.188 1.001-0.313 1.564-0.313 2.19 0 3.941 1.752 3.941 3.941s-1.752 3.941-3.941 3.941z"></path></svg>',
			'spotify'        => '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M25.838 26.369c-0.375 0.626-1.251 0.876-1.877 0.438-5.318-3.191-11.949-3.941-19.769-2.127-0.751 0.125-1.501-0.313-1.689-1.064s0.313-1.501 1.064-1.689c8.571-1.939 15.891-1.126 21.834 2.502 0.688 0.375 0.876 1.251 0.438 1.939zM28.59 20.238c-0.5 0.813-1.564 1.064-2.377 0.563-6.068-3.691-15.265-4.755-22.459-2.628-0.876 0.313-1.877-0.25-2.19-1.126-0.25-0.938 0.25-1.939 1.189-2.19 8.195-2.502 18.33-1.314 25.275 2.94 0.813 0.563 1.126 1.627 0.563 2.44zM31.718 13.107c-0.563 1.001-1.877 1.314-2.878 0.751-7.257-4.317-19.206-4.692-26.151-2.628-1.064 0.375-2.252-0.25-2.628-1.376-0.313-1.126 0.313-2.315 1.439-2.628 7.945-2.44 21.146-1.939 29.529 3.003 1.001 0.626 1.314 1.877 0.688 2.878z"></path></svg>',
			'star'           => '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M30.697 11.417c-0.133-0.534-0.534-0.801-1.068-0.934l-8.542-1.201-3.87-7.741c-0.4-0.934-2.002-0.934-2.402 0l-3.87 7.607-8.408 1.335c-0.534 0-0.934 0.4-1.201 0.934-0.133 0.534 0 1.068 0.4 1.335l6.139 6.006-1.468 8.542c-0.133 0.534 0.133 1.068 0.534 1.335s0.934 0.4 1.468 0.133l7.607-4.004 7.607 4.004c0.267 0 0.4 0 0.667 0s0.534-0.133 0.801-0.267c0.4-0.267 0.667-0.801 0.534-1.335l-1.468-8.542 6.139-6.006c0.4-0.267 0.534-0.801 0.4-1.201zM21.755 17.29c-0.267 0.4-0.4 0.801-0.4 1.201l1.068 6.54-5.872-3.070c-0.4-0.267-0.801-0.267-1.201 0l-5.872 3.070 1.201-6.54c0-0.4-0.133-0.801-0.4-1.201l-4.805-4.538 6.54-0.934c0.4 0 0.801-0.4 1.068-0.667l2.936-6.006 2.936 5.872c0.133 0.4 0.534 0.667 1.068 0.667l6.54 0.934-4.805 4.671z"></path></svg>',
			'tag'            => '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M28.428 12.619l-11.478-11.478c-0.267-0.267-0.534-0.4-0.934-0.4h-13.346c-0.801 0-1.335 0.534-1.335 1.335v13.346c0 0.4 0.133 0.667 0.4 0.934l11.478 11.478c0.801 0.801 1.868 1.201 2.803 1.201 1.068 0 2.002-0.4 2.803-1.201l9.609-9.609c0 0 0 0 0 0 1.602-1.602 1.602-4.004 0-5.605zM26.559 16.356l-9.609 9.609c-0.534 0.534-1.335 0.534-1.868 0l-11.077-11.077v-11.478h11.478l11.077 11.077c0.534 0.534 0.534 1.335 0 1.868zM8.408 7.814c-0.267 0.267-0.4 0.534-0.4 0.934s0.133 0.667 0.4 0.934c0.267 0.267 0.534 0.4 0.934 0.4s0.667-0.133 0.934-0.4c0.267-0.267 0.4-0.534 0.4-0.934s-0.133-0.667-0.4-0.934c-0.534-0.534-1.335-0.534-1.868 0z"></path></svg>',
			'tumblr'         => '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M6.131 13.544v-4.567c1.251-0.375 2.377-1.001 3.253-1.752s1.627-1.689 2.127-2.753c0.563-1.064 0.938-2.44 1.126-4.066h4.567v8.070h7.632v5.067h-7.632v8.195c0 1.877 0.125 3.066 0.313 3.566 0.188 0.563 0.563 0.938 1.064 1.251 0.751 0.438 1.564 0.688 2.44 0.688 1.627 0 3.253-0.563 4.88-1.627v5.067c-1.376 0.626-2.628 1.126-3.754 1.376-1.064 0.25-2.315 0.375-3.566 0.375-1.501 0-2.753-0.188-3.879-0.563s-2.127-0.876-2.878-1.564c-0.813-0.688-1.376-1.439-1.689-2.19s-0.438-1.877-0.438-3.378v-11.198z"></path></svg>',
			'twitch'         => '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M22.96 28.246h-6.256l-4.192 4.192h-4.129v-4.192h-7.695v-22.272l2.127-5.568h28.528v19.519zM28.528 18.487v-15.327h-22.96v20.207h6.256v4.192l4.192-4.192h7.632zM20.895 8.79v8.321h2.753v-8.321zM13.2 17.11h2.815v-8.321h-2.815z"></path></svg>',
			'twitter'        => '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M28.778 9.916c0 0.25 0 0.563 0 0.813 0 8.696-6.631 18.706-18.706 18.706-3.691 0-7.195-1.064-10.072-2.94 0.5 0.063 1.064 0.063 1.564 0.063 3.066 0 5.943-1.001 8.195-2.815-2.878 0-5.318-1.939-6.194-4.567 0.438 0.125 0.813 0.125 1.251 0.125 0.626 0 1.189-0.063 1.752-0.188-3.003-0.626-5.255-3.316-5.255-6.444 0-0.063 0-0.063 0-0.125 0.876 0.501 1.877 0.813 2.94 0.813-1.752-1.126-2.94-3.191-2.94-5.443 0-1.189 0.375-2.315 0.938-3.316 3.191 4.004 8.070 6.631 13.513 6.882-0.125-0.501-0.188-1.001-0.188-1.501 0-3.629 2.94-6.569 6.569-6.569 1.939 0 3.629 0.813 4.817 2.065 1.501-0.313 2.94-0.813 4.192-1.564-0.5 1.501-1.564 2.815-2.878 3.629 1.314-0.188 2.565-0.5 3.754-1.064-0.876 1.314-2.002 2.502-3.253 3.441z"></path></svg>',
			'vimeo'          => '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M20.708 9.541c-0.751 0-1.564 0.188-2.377 0.563 1.564-5.193 4.567-7.695 9.009-7.507 3.316 0.063 4.88 2.19 4.692 6.381-0.188 3.128-2.315 7.382-6.569 12.825-4.317 5.63-8.008 8.508-11.011 8.508-1.877 0-3.441-1.752-4.755-5.193-0.876-3.191-1.752-6.319-2.565-9.509-1.001-3.441-2.002-5.13-3.128-5.13-0.25 0-1.064 0.501-2.502 1.501l-1.501-1.939c1.564-1.439 3.128-2.815 4.692-4.192 2.065-1.814 3.691-2.753 4.755-2.878 2.44-0.25 4.004 1.439 4.567 5.13 0.626 3.941 1.064 6.381 1.314 7.32 0.688 3.253 1.501 4.88 2.377 4.88 0.688 0 1.689-1.064 3.003-3.128 1.314-2.127 2.065-3.754 2.19-4.88 0.188-1.814-0.563-2.753-2.19-2.753z"></path></svg>',
			'vkontakte'      => '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M30.968 22.491c-0.938-1.251-1.752-2.002-2.94-3.128-0.5-0.5-1.189-1.001-1.251-1.689-0.063-0.938 0.813-1.814 1.251-2.44 0.063-0.063 0.125-0.125 0.188-0.188s0.063-0.063 0.125-0.125c0-0.063 0.063-0.063 0.063-0.125 0.063-0.063 0.063-0.125 0.125-0.125 0-0.063 0.063-0.063 0.063-0.063 0.063-0.063 0.063-0.125 0.125-0.188 0.813-1.064 1.627-2.002 2.44-3.253 0.25-0.438 1.064-2.002 0.813-2.628-0.375-0.751-2.315-0.501-3.378-0.501-1.314 0-2.753-0.25-3.691 0.125-0.563 0.188-0.938 1.439-1.251 2.127-0.5 1.064-0.938 1.939-1.439 2.753 0 0.063-0.063 0.125-0.125 0.188 0 0 0 0 0 0 0 0.063-0.063 0.125-0.125 0.188 0 0 0 0 0 0-0.063 0.125-0.125 0.25-0.188 0.313 0 0.063-0.063 0.063-0.063 0.063 0 0.063-0.063 0.063-0.063 0.125-0.063 0.063-0.063 0.063-0.063 0.125-0.063 0-0.063 0.063-0.125 0.125 0 0-0.063 0.063-0.063 0.125-0.063 0-0.063 0.063-0.063 0.063-0.063 0.063-0.125 0.125-0.188 0.25-0.438 0.626-0.876 1.376-1.689 1.501-0.063-0.063-0.063-0.063-0.125-0.063 0 0 0-0.063 0-0.063-0.063 0-0.063 0-0.063 0 0-0.063-0.063-0.063-0.063-0.063s0 0-0.063-0.063c0 0 0 0 0 0s0-0.063-0.063-0.063c0 0 0 0 0 0 0-0.063-0.063-0.063-0.063-0.063s0-0.063 0-0.063c0 0-0.063 0-0.063-0.063 0 0 0 0 0 0 0-0.063 0-0.063-0.063-0.063 0 0 0-0.063 0-0.063s0 0 0-0.063c0 0-0.063 0-0.063 0 0-0.063 0-0.063 0-0.125 0 0 0 0 0 0-0.063-0.063-0.063-0.063-0.063-0.063 0-0.063 0-0.063 0-0.063s0-0.063 0-0.063c0 0 0-0.063-0.063-0.063 0 0 0-0.063 0-0.063s0 0 0-0.063c0 0 0 0 0-0.063 0 0 0 0 0 0 0-0.063 0-0.063-0.063-0.063 0-0.063 0-0.063 0-0.063 0-0.063 0-0.063 0-0.063 0-0.063 0-0.063 0-0.063 0-0.063 0-0.063 0-0.063 0-0.063 0-0.063 0-0.063 0-0.063 0-0.063 0-0.125 0 0 0 0 0 0 0-0.063 0-0.063 0-0.125 0 0 0 0 0 0 0-0.063 0-0.063 0-0.125 0 0-0.063 0-0.063-0.063 0 0 0-0.063 0-0.063s0-0.063 0-0.063c0 0 0-0.063 0-0.063s0-0.063 0-0.063c0 0 0-0.063 0-0.063 0-0.063 0.063-0.063 0.063-0.063 0-0.063 0-0.063 0-0.063 0-0.063 0-0.063 0-0.063 0-0.063 0-0.063 0-0.125 0 0 0 0 0 0 0-0.063 0-0.063 0-0.125 0 0 0 0 0 0 0-0.188 0-0.313 0-0.501 0 0 0-0.063 0-0.063 0-0.125 0-0.313 0.063-0.438 0-1.376 0.438-3.191-0.188-4.254-0.563-0.938-2.69-0.813-4.192-0.813-1.814 0-3.253 0.125-3.816 1.189 0.626 0.438 1.314 0.438 1.627 1.064 0.375 0.688 0.438 2.127 0.375 3.441 0 0.063 0 0.125 0 0.188 0 1.126-0.063 2.94-0.813 3.066s-1.376-1.001-1.814-1.564c-0.063-0.125-0.125-0.188-0.188-0.313 0 0-0.063-0.063-0.063-0.063-0.063-0.063-0.063-0.125-0.125-0.25 0 0-0.063-0.063-0.063-0.063-0.063-0.063-0.125-0.125-0.125-0.188-0.063-0.063-0.063-0.063-0.063-0.125-0.063-0.063-0.125-0.188-0.188-0.25-0.563-0.938-1.126-1.877-1.627-3.066-0.375-0.688-0.813-1.877-1.189-2.065-0.751-0.375-2.252-0.188-3.253-0.188-1.126 0-2.315-0.188-3.003 0.375 0 1.251 0.626 2.315 1.126 3.316 0.688 1.376 1.376 2.69 2.127 3.941 0 0.063 0.063 0.125 0.063 0.188 0.125 0.125 0.188 0.313 0.313 0.438 0 0.063 0.063 0.125 0.063 0.188 0.125 0.125 0.188 0.313 0.313 0.438 0 0.063 0.063 0.125 0.125 0.188 0.125 0.188 0.25 0.375 0.375 0.563 0 0 0 0 0 0 0.125 0.188 0.25 0.438 0.375 0.626 0.063 0.063 0.063 0.125 0.125 0.125 0.125 0.188 0.188 0.313 0.313 0.438 0 0.063 0.063 0.125 0.125 0.188 0.063 0.125 0.188 0.313 0.313 0.438 0 0.063 0.063 0.125 0.125 0.188 0.125 0.188 0.25 0.375 0.438 0.563 1.939 2.44 4.129 4.755 8.446 5.255 1.376 0.125 3.378 0.25 3.816-0.563 0.626-1.001-0.063-3.378 1.314-3.566 0.938-0.188 1.752 1.251 2.315 1.814 0.938 1.064 2.065 2.19 3.691 2.502 0 0 0.063 0 0.125 0 1.439-0.626 5.756 0.688 6.006-1.126 0.063-0.626-0.688-1.627-1.064-2.065z"></path></svg>',
			'x'              => '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M17.884 15.421l7.074-7.074c0.534-0.534 0.534-1.335 0-1.868s-1.335-0.534-1.868 0l-7.074 7.074-7.074-7.074c-0.534-0.534-1.335-0.534-1.868 0s-0.534 1.335 0 1.868l7.074 7.074-7.074 7.074c-0.534 0.534-0.534 1.335 0 1.868 0.267 0.267 0.534 0.4 0.934 0.4s0.667-0.133 0.934-0.4l7.074-7.074 7.074 7.074c0.267 0.267 0.667 0.4 0.934 0.4s0.667-0.133 0.934-0.4c0.534-0.534 0.534-1.335 0-1.868l-7.074-7.074z"></path></svg>',
			'xing'           => '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M7.695 22.866h-4.63c-0.25 0-0.438-0.125-0.563-0.313s-0.125-0.5 0-0.751l4.88-8.633c0 0 0 0 0-0.063l-3.128-5.38c-0.125-0.25-0.125-0.5 0-0.688s0.313-0.313 0.626-0.313h4.63c0.688 0 1.064 0.438 1.251 0.876 0 0 3.191 5.505 3.191 5.505-0.188 0.375-5.005 8.821-5.005 8.821-0.25 0.438-0.563 0.938-1.251 0.938zM29.529 1.408l-10.197 18.080c0 0 0 0.063 0 0.063l6.506 11.887c0.125 0.25 0.125 0.5 0 0.688s-0.313 0.313-0.626 0.313h-4.63c-0.688 0-1.001-0.5-1.251-0.876 0 0-6.569-12.012-6.569-12.074 0.313-0.563 10.26-18.205 10.26-18.205 0.25-0.438 0.563-0.876 1.251-0.876h4.692c0.25 0 0.5 0.125 0.563 0.313 0.125 0.188 0.125 0.438 0 0.688z"></path></svg>',
			'quote'          => '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 123.961 123.961"><path d="M49.8,29.032c3.1-1.3,4.4-5,3-8l-4.9-10.3c-1.4-2.899-4.8-4.2-7.8-2.899c-8.5,3.6-15.8,8.3-21.6,14 C11.4,28.532,6.6,36.232,4,44.732c-2.6,8.601-4,20.3-4,35.2v30.7c0,3.3,2.7,6,6,6h39.3c3.3,0,6-2.7,6-6v-39.3c0-3.301-2.7-6-6-6 H26.5c0.2-10.101,2.6-18.2,7-24.301C37.1,36.133,42.5,32.133,49.8,29.032z"/><path d="M120.4,29.032c3.1-1.3,4.399-5,3-8l-4.9-10.199c-1.4-2.9-4.8-4.2-7.8-2.9c-8.4,3.6-15.601,8.3-21.5,13.9 c-7.101,6.8-12,14.5-14.601,23c-2.6,8.399-3.899,20.1-3.899,35.1v30.7c0,3.3,2.7,6,6,6H116c3.3,0,6-2.7,6-6v-39.3 c0-3.301-2.7-6-6-6H97.1c0.2-10.101,2.601-18.2,7-24.301C107.7,36.133,113.1,32.133,120.4,29.032z"/></svg>',
			'yelp'           => '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M19.027 20.421c-0.54 0.545-0.083 1.535-0.083 1.535l4.065 6.788c0 0 0.668 0.895 1.246 0.895 0.581 0 1.155-0.477 1.155-0.477l3.214-4.594c0 0 0.324-0.58 0.331-1.087 0.012-0.722-1.077-0.92-1.077-0.92l-7.609-2.444c-0 0-0.746-0.198-1.242 0.303zM18.642 17c0.389 0.66 1.463 0.468 1.463 0.468l7.592-2.219c0 0 1.035-0.421 1.182-0.982 0.145-0.561-0.171-1.238-0.171-1.238l-3.628-4.274c0 0-0.314-0.54-0.967-0.595-0.719-0.061-1.162 0.809-1.162 0.809l-4.29 6.75c0 0-0.379 0.672-0.020 1.28zM15.055 14.368c0.894-0.22 1.036-1.519 1.036-1.519l-0.061-10.808c0 0-0.135-1.333-0.734-1.695-0.94-0.57-1.218-0.272-1.487-0.233l-6.303 2.342c0 0-0.617 0.204-0.939 0.719-0.459 0.73 0.466 1.798 0.466 1.798l6.551 8.93c0 0 0.647 0.669 1.47 0.465zM13.498 18.742c0.023-0.834-1.001-1.334-1.001-1.334l-6.775-3.423c0 0-1.004-0.414-1.491-0.126-0.373 0.221-0.704 0.62-0.737 0.973l-0.441 5.432c0 0-0.066 0.941 0.178 1.369 0.345 0.608 1.482 0.184 1.482 0.184l7.909-1.748c0.307-0.207 0.846-0.225 0.876-1.328zM15.465 21.673c-0.679-0.349-1.491 0.373-1.491 0.373l-5.296 5.83c0 0-0.661 0.892-0.493 1.439 0.158 0.513 0.42 0.768 0.791 0.948l5.319 1.679c0 0 0.645 0.134 1.133-0.008 0.693-0.201 0.565-1.286 0.565-1.286l0.12-7.894c-0 0-0.027-0.76-0.648-1.082z"></path></svg>',
			'youtube'        => '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M31.718 10.041c0 0-0.313-2.252-1.251-3.191-1.251-1.314-2.628-1.314-3.253-1.376-4.442-0.313-11.198-0.313-11.198-0.313v0c0 0-6.757 0-11.198 0.313-0.626 0.063-2.002 0.063-3.253 1.376-0.938 0.938-1.251 3.191-1.251 3.191s-0.313 2.565-0.313 5.13v2.44c0 2.628 0.313 5.193 0.313 5.193s0.313 2.19 1.251 3.191c1.251 1.251 2.878 1.251 3.566 1.376 2.565 0.25 10.886 0.313 10.886 0.313s6.757 0 11.198-0.313c0.626-0.063 2.002-0.125 3.253-1.376 0.938-1.001 1.251-3.191 1.251-3.191s0.313-2.565 0.313-5.193v-2.44c0-2.565-0.313-5.13-0.313-5.13zM20.332 16.673v0l-7.57 3.941v-9.009l8.633 4.504z"></path></svg>',
			'whatsapp'       => '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 32 32"><path d="M27.281 4.65c-2.994-3-6.975-4.65-11.219-4.65-8.738 0-15.85 7.112-15.85 15.856 0 2.794 0.731 5.525 2.119 7.925l-2.25 8.219 8.406-2.206c2.319 1.262 4.925 1.931 7.575 1.931h0.006c0 0 0 0 0 0 8.738 0 15.856-7.113 15.856-15.856 0-4.238-1.65-8.219-4.644-11.219zM16.069 29.050v0c-2.369 0-4.688-0.637-6.713-1.837l-0.481-0.288-4.987 1.306 1.331-4.863-0.313-0.5c-1.325-2.094-2.019-4.519-2.019-7.012 0-7.269 5.912-13.181 13.188-13.181 3.519 0 6.831 1.375 9.319 3.862 2.488 2.494 3.856 5.8 3.856 9.325-0.006 7.275-5.919 13.188-13.181 13.188zM23.294 19.175c-0.394-0.2-2.344-1.156-2.706-1.288s-0.625-0.2-0.894 0.2c-0.262 0.394-1.025 1.288-1.256 1.556-0.231 0.262-0.462 0.3-0.856 0.1s-1.675-0.619-3.188-1.969c-1.175-1.050-1.975-2.35-2.206-2.744s-0.025-0.613 0.175-0.806c0.181-0.175 0.394-0.463 0.594-0.694s0.262-0.394 0.394-0.662c0.131-0.262 0.069-0.494-0.031-0.694s-0.894-2.15-1.219-2.944c-0.319-0.775-0.65-0.669-0.894-0.681-0.231-0.012-0.494-0.012-0.756-0.012s-0.694 0.1-1.056 0.494c-0.363 0.394-1.387 1.356-1.387 3.306s1.419 3.831 1.619 4.1c0.2 0.262 2.794 4.269 6.769 5.981 0.944 0.406 1.681 0.65 2.256 0.837 0.95 0.3 1.813 0.256 2.494 0.156 0.762-0.113 2.344-0.956 2.675-1.881s0.331-1.719 0.231-1.881c-0.094-0.175-0.356-0.275-0.756-0.475z"></path></svg>',
		);
	}
endif;
